﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Numerics;
using System.Reflection;
using System.Text;

namespace algocs_reflection
{
    class Program
    {
        static void Main(string[] args)
        {
            // переменная 'x' имеет тип object, но хранит ссылку на объекта типа ClassX
            // метод GetType возвращает реальный тип объекта
            object x = new ClassX("Mister X");
            Type type = x.GetType();

            // другой способ получить тип - с помощью оператора typeof
            Type type1 = typeof(ClassX);
            if (type == type1)
                Console.WriteLine("x.GetType() == typeof(ClassX)");

            // мы можем получить информацию о членах класса
            // (по умолчанию - публичных, в т.ч. о статических и унаследованных)
            Console.WriteLine("-- GetMembers()");
            MemberInfo[] members = type.GetMembers();
            foreach (var m in members)
                Console.WriteLine(m);
            Console.WriteLine();

            // но можем добраться и до приватных
            Console.WriteLine("-- GetMembers(NonPublic | Instance)");
            members = type.GetMembers(BindingFlags.NonPublic | BindingFlags.Instance);
            foreach (var m in members)
                Console.WriteLine(m);
            Console.WriteLine();

            // члены класса - не статические и не унаследованные
            Console.WriteLine("-- GetMembers(Public | DeclaredOnly | Instance)");
            members = type.GetMembers(BindingFlags.Public | BindingFlags.DeclaredOnly | BindingFlags.Instance);
            foreach (var m in members)
                Console.WriteLine(m);
            Console.WriteLine();

            // пока что мы работали с типом, определенном в нашем проекте (сборке = assembly)
            // или в сборке, на которую есть ссылка (см. раздел References в дереве проекта).
            // но также можно загрузить другую сборку динамически.
            // (для того, чтобы работал следующий код, скопируйте algocs_stack.exe в директорию,
            // откуда запускается algocs_reflection.exe; вообще обычно сборки с библиотекой классов
            // называются *.dll и это выглядит более логичным, но и сборку *.exe можно использовать
            // таким же образом, чем мы здесь и пользуемся).

            // определяем путь
            string path = Assembly.GetExecutingAssembly().Location;
            path = Path.GetDirectoryName(path) + @"\algocs_stack.exe";

            // загружаем внешнюю сборку
            var assembly = Assembly.LoadFile(path);
            if (assembly == null)
            {
                Console.WriteLine("Failed to load external assembly");
                return;
            }

            // какие типы доступны в этой сборке?
            var types = assembly.GetTypes();
            foreach (var t in types)
                Console.WriteLine(t);

            // загружаем нужный нам тип (требуется указать fully qualified имя - с указанием namespace)
            var type2 = assembly.GetType("algocs_stack.Stack1");

            // создаем объект этого типа (будет использован конструктор без параметров)
            object stack = Activator.CreateInstance(type2);

            // получаем описатель метода Push и вызываем этот метод
            MethodInfo push = type2.GetMethod("Push");
            push.Invoke(stack, new object[] { 5 });
            push.Invoke(stack, new object[] { 7 });

            // получаем описатель метода Pop и вызываем этот метод
            MethodInfo pop = type2.GetMethod("Pop");
            object r1 = pop.Invoke(stack, new object[] { });
            Console.WriteLine(r1);
            object r2 = pop.Invoke(stack, new object[] { });
            Console.WriteLine(r2);

            // РЕЗЮМЕ
            // Объекты в управляемых языках содержат ссылку на некий объект-описание их типа.
            // Это позволяет runtime-среде гарантировать безопасность выполнения кода
            // (невозможно использование ссылки на некоторый объект как ссылки на объект другого типа).
            // Механизм reflection позволяет и нам получать информацию о типе и вызывать методы
            // (менять значения полей и свойств). Разумеется вызов метода с помощью
            // reflection пишется сложнее и работает медленнее обычного вызова, но определенные техники
            // (в т.ч. популярные в мире бизнес-приложений) могут быть реализованы только с помощью reflection:
            // - dependency injection
            // - конфигурирование приложения с помощью xml-файлов: можно задавать какие классы использовать и с какими параметрами
            // - автоматическое сохранение / восстановление свойств объектов, сериализация
            // - приложения, реализующие концепцию плагинов
        }
    }
}
